/***************************************************************************
* JVerbnet v1.0.0
* Copyright (c) 2012 Massachusetts Institute of Technology
* 
* JVerbnet is distributed under the terms of the Creative Commons 
* Attribution 3.0 Unported License, which means it may be freely used for 
* all purposes, as long as proper acknowledgment is made.  See the license 
* file included with this distribution for more details.
****************************************************************************/

package edu.mit.jverbnet.data.syntax;

import static edu.mit.jverbnet.util.Checks.NotNull;

import edu.mit.jverbnet.data.selection.IRestrType;
import edu.mit.jverbnet.data.selection.ISelRestrictions;
import edu.mit.jverbnet.data.selection.SelRestrictions;
import edu.mit.jverbnet.data.syntax.SyntaxArgType.VALUE_RULE;

/** 
 * Default implementation of {@link ISyntaxArgDesc}.
 *
 * @author Mark A. Finlayson
 * @version 1.0.0
 * @since JVerbnet 1.0.0
 */
public class SyntaxArgDesc implements ISyntaxArgDesc {
	
	// unchanging fields
	private final SyntaxArgType type;
	private final String value;
	private final INounPhraseType npType;
	private final ISelRestrictions<? extends IRestrType> selRestrs;

	/**
	 * Creates a new syntax argument description with the specified parameters.
	 * May be used to create all syntax arguments other than those of the
	 * {@link SyntaxArgType#NP} type.
	 * 
	 * @param type
	 *            the type; may not be <code>null</code> or
	 *            {@link SyntaxArgType#NP}
	 * @param value
	 *            the value; must meet the restrictions of the type or an
	 *            exception is thrown; see
	 *            {@link VALUE_RULE#checkValue(String)}.
	 * @param selRestrs
	 *            the selectional restrictions for this argument; may be
	 *            <code>null</code>
	 * @since JVerbnet 1.0.0
	 */
	public SyntaxArgDesc(SyntaxArgType type, String value, ISelRestrictions<?  extends IRestrType> selRestrs){
		this(type, value, null, selRestrs);
	}
	
	/**
	 * Creates a new syntax argument description of the {@link SyntaxArgType#NP}
	 * type with the specified parameters.
	 * 
	 * @param type
	 *            the noun phrase type
	 * @param value
	 *            the value; must meet the restrictions of the type or an
	 *            exception is thrown; see
	 *            {@link VALUE_RULE#checkValue(String)}.
	 * @param selRestrs
	 *            the selectional restrictions for this argument; may be
	 *            <code>null</code>
	 * @since JVerbnet 1.0.0
	 */
	public SyntaxArgDesc(INounPhraseType type, String value, ISelRestrictions<? extends IRestrType> selRestrs){
		this(SyntaxArgType.NP, value, type, selRestrs);
	}
	
	/**
	 * Creates a new syntax argument description with full control.
	 * 
	 * @param type
	 *            the type; may not be <code>null</code>
	 * @param value
	 *            the value; must meet the restrictions of the type or an
	 *            exception is thrown; see
	 *            {@link VALUE_RULE#checkValue(String)}.
	 * @param npType
	 *            must be non-<code>null</code> if the type is
	 *            {@link SyntaxArgType#NP}; otherwise must be <code>null</code>
	 * @param selRestrs
	 *            the selectional restrictions for this argument; may be
	 *            <code>null</code>
	 * @throws NullPointerException
	 *             if type is <code>null</code>; or if the type requires the
	 *             value not to be <code>null</code> and the value is
	 *             <code>null</code>
	 * @since JVerbnet 1.0.0
	 */
	public SyntaxArgDesc(SyntaxArgType type, String value, INounPhraseType npType, ISelRestrictions<? extends IRestrType> selRestrs){
		// check arguments
		NotNull.check("type", type);
		value = type.getValueRule().checkValue(value);
		if(selRestrs == null)
			selRestrs = SelRestrictions.emptyRestrictions();
		
		// check type/npType compatibility
		if(type == SyntaxArgType.NP && npType == null)
			throw new IllegalArgumentException("Must specific npType if argument type is NP");
		if(type != SyntaxArgType.NP && npType != null)
			throw new IllegalArgumentException("May not specific npType if argument type is not NP");
		
		// assign fields
		this.type = type;
		this.value = value;
		this.npType = npType;
		this.selRestrs = selRestrs;
	}

	/* 
	 * (non-Javadoc) 
	 *
	 * @see edu.mit.jverbnet.data.syntax.ISyntaxArgDesc#getType()
	 */
	public SyntaxArgType getType() {
		return type;
	}

	/* 
	 * (non-Javadoc) 
	 *
	 * @see edu.mit.jverbnet.data.syntax.ISyntaxArgDesc#getValue()
	 */
	public String getValue() {
		return value;
	}

	/* 
	 * (non-Javadoc) 
	 *
	 * @see edu.mit.jverbnet.data.syntax.ISyntaxArgDesc#getNounPhraseType()
	 */
	public INounPhraseType getNounPhraseType() {
		return npType;
	}

	/* 
	 * (non-Javadoc) 
	 *
	 * @see edu.mit.jverbnet.data.syntax.ISyntaxArgDesc#getSelRestrictions()
	 */
	public ISelRestrictions<? extends IRestrType> getSelRestrictions() {
		return selRestrs;
	}

}
